<?php
/**
 * Plugin Name: Limit Auto Translator (WPML + GPT + DeepL) by LiMiT Agency
 * Plugin URI:  https://www.limit.agency/
 * Description: Automate multilingual publishing with WPML. Translates Title/Excerpt/Content on save using GPT first, DeepL as fallback. Queued via WP-Cron for scale. Optional footer credit link to drive traffic to LiMiT Agency. Settings under Settings → Limit Auto Translator.
 * Version:     1.3.0
 * Author:      LiMiT Agency
 * Author URI:  https://www.limit.agency/professional-web-development-service/
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: limit-auto-translator-wpml-gpt-deepl-by-limit-agency
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

define('LIMIT_AT_VER', '1.3.0');
define('LIMIT_AT_SLUG', 'limit-auto-translator-wpml-gpt-deepl-by-limit-agency'); // must match plugin slug
define('LIMIT_AT_OPT',  'limit_at_options');
define('LIMIT_AT_CRON', 'limit_at_worker_event');

/**
 * Default Options
 */
function limit_at_default_options() {
    return array(
        'source_lang'   => 'en',
        'target_langs'  => array('fr','es','ro'),
        'post_types'    => array('post','page'),
        'openai_key'    => '',        // can be left blank to use env
        'openai_model'  => 'gpt-4.1-mini',
        'deepl_key'     => '',        // optional; fallback
        'mode'          => 'queue',   // 'queue' or 'immediate'
        'credit_enable' => 1,         // append credit link to translated posts
        'credit_text'   => 'Translated automatically by LiMiT Agency',
        'credit_url'    => 'https://www.limit.agency/',
        'queue'         => array(),   // internal: list of post IDs pending
        'rate_limit'    => 3,         // items per run
        'timeout'       => 60,        // remote timeout seconds
        'debug'         => 0,         // show admin notices/log
    );
}

/**
 * Activation / Deactivation
 */
register_activation_hook(__FILE__, function() {
    $opt = get_option(LIMIT_AT_OPT, array());
    $opt = wp_parse_args($opt, limit_at_default_options());
    update_option(LIMIT_AT_OPT, $opt, false);
    if (!wp_next_scheduled(LIMIT_AT_CRON)) {
        wp_schedule_event(time()+60, 'minute', LIMIT_AT_CRON);
    }
});

register_deactivation_hook(__FILE__, function() {
    $ts = wp_next_scheduled(LIMIT_AT_CRON);
    if ($ts) wp_unschedule_event($ts, LIMIT_AT_CRON);
});

/**
 * Cron Schedule: Every Minute
 */
add_filter('cron_schedules', function($s) {
    if (!isset($s['minute'])) {
        $s['minute'] = array(
            'interval' => 60,
            'display'  => __('Every Minute', 'limit-auto-translator-wpml-gpt-deepl-by-limit-agency')
        );
    }
    return $s;
});

/**
 * Settings Page
 */
add_action('admin_menu', function() {
    add_options_page(
        'Limit Auto Translator',
        'Limit Auto Translator',
        'manage_options',
        LIMIT_AT_SLUG,
        'limit_at_render_settings'
    );
});

add_action('admin_init', function() {
    register_setting(LIMIT_AT_OPT, LIMIT_AT_OPT, 'limit_at_sanitize');
});

function limit_at_render_settings() {
    if (!current_user_can('manage_options')) return;
    $opt = wp_parse_args(get_option(LIMIT_AT_OPT, array()), limit_at_default_options()); ?>
    <div class="wrap">
        <h1>Limit Auto Translator</h1>
        <p>Built by <a href="https://www.limit.agency/" target="_blank" rel="noopener">LiMiT Agency</a> — automate multilingual publishing with WPML, GPT, and DeepL.</p>

        <form method="post" action="options.php">
            <?php settings_fields(LIMIT_AT_OPT); ?>

            <table class="form-table" role="presentation">
                <tr><th scope="row">Source Language</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[source_lang]" value="<?php echo esc_attr($opt['source_lang']); ?>" class="regular-text">
                        <p class="description">WPML code (e.g., en)</p></td>
                </tr>

                <tr><th scope="row">Target Languages</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[target_langs]" value="<?php echo esc_attr(implode(',', (array)$opt['target_langs'])); ?>" class="regular-text">
                        <p class="description">Comma-separated WPML codes, e.g. <code>fr,es,ro</code></p></td>
                </tr>

                <tr><th scope="row">Post Types</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[post_types]" value="<?php echo esc_attr(implode(',', (array)$opt['post_types'])); ?>" class="regular-text">
                        <p class="description">Comma-separated: <code>post,page,product</code></p></td>
                </tr>

                <tr><th scope="row">OpenAI API Key</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[openai_key]" value="<?php echo esc_attr($opt['openai_key']); ?>" class="regular-text" placeholder="Leave blank to use OPENAI_API_KEY env"></td>
                </tr>

                <tr><th scope="row">OpenAI Model</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[openai_model]" value="<?php echo esc_attr($opt['openai_model']); ?>" class="regular-text"></td>
                </tr>

                <tr><th scope="row">DeepL API Key (optional)</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[deepl_key]" value="<?php echo esc_attr($opt['deepl_key']); ?>" class="regular-text" placeholder="Leave blank to disable DeepL fallback"></td>
                </tr>

                <tr><th scope="row">Run Mode</th>
                    <td>
                        <select name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[mode]">
                            <option value="queue" <?php selected($opt['mode'], 'queue'); ?>>Queue (WP-Cron)</option>
                            <option value="immediate" <?php selected($opt['mode'], 'immediate'); ?>>Immediate (on save)</option>
                        </select>
                        <p class="description">Queue recommended for stability and scale.</p>
                    </td>
                </tr>

                <tr><th scope="row">Rate Limit (per minute)</th>
                    <td><input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[rate_limit]" value="<?php echo esc_attr($opt['rate_limit']); ?>" class="small-text" type="number" min="1" max="50"></td>
                </tr>

                <tr><th scope="row">Append Credit Link</th>
                    <td>
                        <label><input type="checkbox" name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[credit_enable]" value="1" <?php checked($opt['credit_enable'], 1); ?>> Enable</label>
                        <p class="description">Adds a footer credit on translated posts linking to LiMiT Agency.</p>
                        <p>
                            Text: <input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[credit_text]" value="<?php echo esc_attr($opt['credit_text']); ?>" class="regular-text">
                            <br>URL: <input name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[credit_url]" value="<?php echo esc_url($opt['credit_url']); ?>" class="regular-text">
                        </p>
                    </td>
                </tr>

                <tr><th scope="row">Debug</th>
                    <td><label><input type="checkbox" name="<?php echo esc_attr(LIMIT_AT_OPT); ?>[debug]" value="1" <?php checked($opt['debug'], 1); ?>> Show admin notices & basic logs</label></td>
                </tr>
            </table>

            <?php submit_button('Save Settings'); ?>
        </form>

        <hr>
        <p><strong>Need help or a custom build?</strong> Talk to us at <a href="https://www.limit.agency/contact/" target="_blank" rel="noopener">LiMiT Agency</a>.</p>
    </div>
<?php }

function limit_at_sanitize($in) {
    $defs = limit_at_default_options();
    $out  = wp_parse_args((array)$in, $defs);
    $out['source_lang']   = sanitize_text_field($out['source_lang']);
    $out['target_langs']  = array_filter(array_map('sanitize_text_field', array_map('trim', is_array($out['target_langs']) ? $out['target_langs'] : explode(',', $out['target_langs']))));
    $out['post_types']    = array_filter(array_map('sanitize_key', array_map('trim', is_array($out['post_types']) ? $out['post_types'] : explode(',', $out['post_types']))));
    $out['openai_key']    = trim($out['openai_key']);
    $out['openai_model']  = sanitize_text_field($out['openai_model']);
    $out['deepl_key']     = trim($out['deepl_key']);
    $out['mode']          = in_array($out['mode'], array('queue','immediate'), true) ? $out['mode'] : 'queue';
    $out['credit_enable'] = !empty($out['credit_enable']) ? 1 : 0;
    $out['credit_text']   = wp_kses_post($out['credit_text']);
    $out['credit_url']    = esc_url_raw($out['credit_url']);
    $out['rate_limit']    = max(1, intval($out['rate_limit']));
    $out['timeout']       = max(10, intval($out['timeout']));
    $out['debug']         = !empty($out['debug']) ? 1 : 0;
    if (!is_array($out['queue'])) $out['queue'] = array();
    return $out;
}

/**
 * Plugin action links
 */
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $links[] = '<a href="' . esc_url(admin_url('options-general.php?page=' . LIMIT_AT_SLUG)) . '">Settings</a>';
    $links[] = '<a href="https://www.limit.agency/" target="_blank" rel="noopener">LiMiT Agency</a>';
    return $links;
});

/**
 * Utils
 */
function limit_at_opt() {
    return wp_parse_args(get_option(LIMIT_AT_OPT, array()), limit_at_default_options());
}
function limit_at_env($k) {
    $v = getenv($k);
    return $v ? $v : '';
}
function limit_at_log($msg) {
    $opt = limit_at_opt();
    if (!$opt['debug']) return;
    error_log('[LimitAT] ' . (is_string($msg) ? $msg : wp_json_encode($msg)));
    add_action('admin_notices', function() use ($msg) {
        echo '<div class="notice notice-info is-dismissible"><p><strong>Limit Auto Translator:</strong> ' . esc_html(is_string($msg) ? $msg : wp_json_encode($msg)) . '</p></div>';
    });
}

/**
 * Enqueue or process on save
 */
add_action('save_post', function($post_id, $post, $update) {
    $opt = limit_at_opt();
    if (wp_is_post_revision($post_id) || (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)) return;
    if (!in_array(get_post_status($post_id), array('publish','pending','draft'), true)) return;
    if (!in_array(get_post_type($post_id), (array)$opt['post_types'], true)) return;

    // Get WPML language
    $lang = apply_filters('wpml_element_language_code', null, array(
        'element_id'   => $post_id,
        'element_type' => 'post_' . get_post_type($post_id),
    ));
    if ($lang !== $opt['source_lang']) return;

    if ($opt['mode'] === 'immediate') {
        limit_at_process_post($post_id);
    } else {
        limit_at_enqueue($post_id);
    }
}, 20, 3);

function limit_at_enqueue($post_id) {
    $opt = limit_at_opt();
    if (!in_array($post_id, $opt['queue'], true)) {
        $opt['queue'][] = $post_id;
        update_option(LIMIT_AT_OPT, $opt, false);
        limit_at_log("Enqueued post $post_id");
    }
}

/**
 * Worker (WP-Cron)
 */
add_action(LIMIT_AT_CRON, function() {
    $opt = limit_at_opt();
    $rate = $opt['rate_limit'];
    $queue = (array)$opt['queue'];
    if (empty($queue)) return;

    $processed = 0;
    while ($processed < $rate && !empty($queue)) {
        $post_id = array_shift($queue);
        limit_at_process_post($post_id);
        $processed++;
    }
    $opt['queue'] = $queue;
    update_option(LIMIT_AT_OPT, $opt, false);
});

/**
 * Core Processor
 */
function limit_at_process_post($post_id) {
    $opt = limit_at_opt();
    $ptype = get_post_type($post_id);
    $src_lang = $opt['source_lang'];

    // Lock to prevent recursion
    if (get_post_meta($post_id, '_limit_at_lock', true)) return;
    update_post_meta($post_id, '_limit_at_lock', 1);

    // Confirm source language
    $lang = apply_filters('wpml_element_language_code', null, array(
        'element_id'   => $post_id,
        'element_type' => 'post_' . $ptype,
    ));
    if ($lang !== $src_lang) {
        delete_post_meta($post_id, '_limit_at_lock');
        return;
    }

    $src_title   = get_the_title($post_id);
    $src_excerpt = get_post_field('post_excerpt', $post_id);
    $src_content = get_post_field('post_content', $post_id);
    $trid = apply_filters('wpml_element_trid', null, $post_id, 'post_' . $ptype);

    foreach ((array)$opt['target_langs'] as $tgt) {
        // Skip existing translations
        $existing = apply_filters('wpml_object_id', $post_id, 'post_' . $ptype, false, $tgt);
        if ($existing) continue;

        // Translate
        $title_tr   = limit_at_translate_block($src_title,   limit_at_lang_label($tgt), limit_at_deepl_code($tgt), 'plain', $opt);
        $excerpt_tr = limit_at_translate_block($src_excerpt, limit_at_lang_label($tgt), limit_at_deepl_code($tgt), 'html',  $opt);
        $content_tr = limit_at_translate_block($src_content, limit_at_lang_label($tgt), limit_at_deepl_code($tgt), 'html',  $opt);

        // Insert post
        $new_id = wp_insert_post(array(
            'post_type'    => $ptype,
            'post_status'  => get_post_status($post_id),
            'post_title'   => $title_tr,
            'post_excerpt' => $excerpt_tr,
            'post_content' => $content_tr,
            'post_author'  => get_post_field('post_author', $post_id),
            'post_name'    => sanitize_title(get_post_field('post_name', $post_id) . '-' . $tgt),
        ));

        if ($new_id && !is_wp_error($new_id)) {
            // Link language
            do_action('wpml_set_element_language_details', array(
                'element_id'           => $new_id,
                'element_type'         => 'post_' . $ptype,
                'trid'                 => $trid,
                'language_code'        => $tgt,
                'source_language_code' => $src_lang,
            ));
            // Copy terms
            limit_at_copy_terms($post_id, $new_id);
            // Append credit footer if enabled
            if (!empty($opt['credit_enable'])) {
                add_post_meta($new_id, '_limit_at_credit', 1, true);
            }
            limit_at_log("Created translation $new_id ($tgt) for post $post_id");
        }
    }

    delete_post_meta($post_id, '_limit_at_lock');
}

function limit_at_lang_label($wpml) {
    $map = array(
        'fr' => 'French', 'es' => 'Spanish', 'ro' => 'Romanian',
        'de' => 'German', 'it' => 'Italian', 'pt' => 'Portuguese', 'nl' => 'Dutch'
    );
    return isset($map[$wpml]) ? $map[$wpml] : strtoupper($wpml);
}
function limit_at_deepl_code($wpml) {
    return strtoupper($wpml); // DeepL accepts FR/ES/RO/DE/IT/PT/NL etc.
}

function limit_at_copy_terms($src_id, $dst_id) {
    $taxes = get_object_taxonomies(get_post_type($src_id));
    foreach ($taxes as $tax) {
        $terms = wp_get_object_terms($src_id, $tax, array('fields' => 'ids'));
        if (!is_wp_error($terms)) {
            wp_set_object_terms($dst_id, $terms, $tax, false);
        }
    }
}

/**
 * Credit Footer
 */
add_filter('the_content', function($content) {
    if (is_admin()) return $content;
    if (!in_the_loop() || !is_singular()) return $content;

    global $post;
    $opt = limit_at_opt();
    if (empty($opt['credit_enable'])) return $content;

    // Use WPML to detect language; only add on non-source languages where meta is set
    $lang = function_exists('apply_filters') ? apply_filters('wpml_current_language', null) : null;
    $is_translated = get_post_meta($post->ID, '_limit_at_credit', true);

    if ($is_translated && $lang && $lang !== $opt['source_lang']) {
        $html = '<div class="limit-at-credit" style="margin-top:24px;padding-top:16px;border-top:1px solid #e5e7eb;font-size:13px;opacity:.85">'
              . esc_html($opt['credit_text'])
              . ' — <a href="' . esc_url($opt['credit_url']) . '" target="_blank" rel="noopener">LiMiT Agency</a></div>';
        $html = apply_filters('limit_at_credit_html', $html, $post->ID, $lang);
        return $content . $html;
    }
    return $content;
}, 20);

/**
 * Translation Pipeline
 */
function limit_at_translate_block($text, $tgt_label, $deepl_code, $mode, $opt) {
    if (trim((string)$text) === '') return $text;

    list($masked, $map) = limit_at_mask($text);

    // GPT first
    $out = limit_at_openai($masked, $tgt_label, $mode, $opt);
    if ($out) {
        $out = limit_at_unmask($out, $map);
        if (limit_at_validate($text, $out, $mode)) return $out;
    }
    // Retry once
    $out2 = limit_at_openai($masked, $tgt_label, $mode, $opt);
    if ($out2) {
        $out2 = limit_at_unmask($out2, $map);
        if (limit_at_validate($text, $out2, $mode)) return $out2;
    }
    // DeepL fallback
    if (!empty($opt['deepl_key'])) {
        $dl = limit_at_deepl($masked, $deepl_code, $mode, $opt);
        if ($dl) return limit_at_unmask($dl, $map);
    }
    return $text;
}

function limit_at_mask($text) {
    $map = array(); $i = 0;
    $pattern = '/('
        . 'https?:\/\/[^\s"\'<>\)]+'
        . '|\[[a-zA-Z0-9_\-]+(?:\s+[^\]]+)?\]'
        . '|\{[A-Z0-9_]+\}'
        . '|%[a-zA-Z]'
        . '|`[^`]+`'
    . ')/';
    $masked = preg_replace_callback($pattern, function($m) use (&$map,&$i) {
        $key = '__PH_' . (++$i) . '__'; $map[$key] = $m[0]; return $key;
    }, $text);
    return array($masked, $map);
}
function limit_at_unmask($text, $map) {
    foreach ($map as $k => $v) $text = str_replace($k, $v, $text);
    return $text;
}
function limit_at_validate($src, $out, $mode) {
    if ($mode === 'plain') return (mb_strlen(trim($out)) > 0);
    $tag_re = '/<\/?([a-zA-Z0-9]+)(\s+[^>]*)?>/';
    preg_match_all($tag_re, $src, $src_tags);
    preg_match_all($tag_re, $out, $out_tags);
    $ok_len  = count($out_tags[0]) >= max(1, floor(count($src_tags[0]) * 0.7));
    $ok_root = true;
    if (!empty($src_tags[1]) && !empty($out_tags[1])) {
        $ok_root = (strtolower($src_tags[1][0]) === strtolower($out_tags[1][0]));
    }
    return ($ok_len && $ok_root);
}

/**
 * OpenAI & DeepL
 */
function limit_at_openai($text, $tgt_label, $mode, $opt) {
    $key   = $opt['openai_key'] ?: limit_at_env('OPENAI_API_KEY');
    if (!$key) return '';
    $sys = "You are a professional web translator. Output " . ($mode === 'html' ? "valid HTML" : "plain text")
        . " only. Preserve tags, attributes, structure, formatting, URLs, and tokens exactly. "
        . "Translate visible text from English to {$tgt_label}. Do not add or remove tags.";
    $payload = array(
        'model'     => $opt['openai_model'],
        'messages'  => array(
            array('role'=>'system','content'=>$sys),
            array('role'=>'user','content'=>$text),
        ),
        'temperature'=> 0.2
    );
    $res = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
        'timeout' => max(30, intval($opt['timeout'])),
        'headers' => array(
            'Authorization' => 'Bearer ' . $key,
            'Content-Type'  => 'application/json',
        ),
        'body' => wp_json_encode($payload),
    ));
    if (is_wp_error($res)) { limit_at_log($res->get_error_message()); return ''; }
    if ((int)wp_remote_retrieve_response_code($res) >= 300) return '';
    $data = json_decode(wp_remote_retrieve_body($res), true);
    return !empty($data['choices'][0]['message']['content']) ? trim($data['choices'][0]['message']['content']) : '';
}

function limit_at_deepl($text, $tgt_code, $mode, $opt) {
    $key = $opt['deepl_key'];
    if (!$key) return '';
    $args = array(
        'auth_key'            => $key,
        'text'                => $text,
        'target_lang'         => strtoupper($tgt_code),
        'preserve_formatting' => '1',
    );
    if ($mode === 'html') {
        $args['tag_handling']      = 'html';
        $args['outline_detection'] = '1';
    }
    $res = wp_remote_post('https://api-free.deepl.com/v2/translate', array(
        'timeout' => max(30, intval($opt['timeout'])),
        'body'    => $args,
    ));
    if (is_wp_error($res)) { limit_at_log($res->get_error_message()); return ''; }
    if ((int)wp_remote_retrieve_response_code($res) >= 300) return '';
    $data = json_decode(wp_remote_retrieve_body($res), true);
    return !empty($data['translations'][0]['text']) ? $data['translations'][0]['text'] : '';
}

/**
 * WP-CLI
 */
if (defined('WP_CLI') && WP_CLI) {
    WP_CLI::add_command('limit-at', function($args, $assoc) {
        if (!empty($assoc['post'])) {
            $id = intval($assoc['post']);
            limit_at_process_post($id);
            WP_CLI::success("Processed post $id.");
            return;
        }
        if (!empty($assoc['enqueue'])) {
            $q = array_map('intval', explode(',', $assoc['enqueue']));
            foreach ($q as $id) limit_at_enqueue($id);
            WP_CLI::success("Enqueued: " . implode(',', $q));
            return;
        }
        WP_CLI::log("Usage:");
        WP_CLI::log("  wp limit-at --post=123        # translate a single post now");
        WP_CLI::log("  wp limit-at --enqueue=1,2,3   # enqueue multiple");
    });
}

/**
 * Admin footer branding on settings page
 */
add_filter('admin_footer_text', function($text) {
    if (get_current_screen() && get_current_screen()->id === 'settings_page_' . LIMIT_AT_SLUG) {
        $text = 'Built with ❤️ by <a href="https://www.limit.agency/" target="_blank" rel="noopener">LiMiT Agency</a>.';
    }
    return $text;
});
